---
title: Bind to Workers API · Cloudflare Images docs
description: A binding connects your Worker to external resources on the
  Developer Platform, like Images, R2 buckets, or KV Namespaces.
lastUpdated: 2025-07-24T15:34:32.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/images/transform-images/bindings/
  md: https://developers.cloudflare.com/images/transform-images/bindings/index.md
---

A [binding](https://developers.cloudflare.com/workers/runtime-apis/bindings/) connects your [Worker](https://developers.cloudflare.com/workers/) to external resources on the Developer Platform, like [Images](https://developers.cloudflare.com/images/transform-images/transform-via-workers/), [R2 buckets](https://developers.cloudflare.com/r2/buckets/), or [KV Namespaces](https://developers.cloudflare.com/kv/concepts/kv-namespaces/).

You can bind the Images API to your Worker to transform, resize, and encode images without requiring them to be accessible through a URL.

For example, when you allow Workers to interact with Images, you can:

* Transform an image, then upload the output image directly into R2 without serving to the browser.
* Optimize an image stored in R2 by passing the blob of bytes representing the image, instead of fetching the public URL for the image.
* Resize an image, overlay the output over a second image as a watermark, then resize this output into a final result.

Bindings can be configured in the Cloudflare dashboard for your Worker or in the `wrangler.toml` file in your project's directory.

## Setup

The Images binding is enabled on a per-Worker basis.

You can define variables in the `wrangler.toml` file of your Worker project's directory. These variables are bound to external resources at runtime, and you can then interact with them through this variable.

To bind Images to your Worker, add the following to the end of your `wrangler.toml` file:

* wrangler.jsonc

  ```jsonc
  {
    "images": {
      "binding": "IMAGES", // i.e. available in your Worker on env.IMAGES
    },
  }
  ```

* wrangler.toml

  ```toml
  [images]
  binding = "IMAGES"
  ```

Within your Worker code, you can interact with this binding by using `env.IMAGES.input()` to build an object that can manipulate the image (passed as a `ReadableStream`).

## Methods

### `.transform()`

* Defines how an image should be optimized and manipulated through [parameters](https://developers.cloudflare.com/images/transform-images/transform-via-workers/#fetch-options) such as `width`, `height`, and `blur`.

### `.draw()`

* Allows [drawing an image](https://developers.cloudflare.com/images/transform-images/draw-overlays/) over another image.
* The drawn image can be a stream, or another image returned from `.input()` that has been manipulated.
* The overlaid image can be manipulated using `opacity`, `repeat`, `top`, `left`, `bottom`, and `right`. To apply other parameters, you can pass a child `.transform()` function inside this method.

For example, to draw a resized watermark on an image:

```ts
// Fetch the watermark from Workers Assets, R2, KV etc
const watermark: ReadableStream = ...


// Fetch the main image
const image: ReadableStream = ...


const response = (
  await env.IMAGES.input(image)
    .draw(
        env.IMAGES.input(watermark)
          .transform({ width: 32, height: 32}),
        { bottom: 32, right: 32 }
    )
    .output({ format: "image/avif" })
).response()


return response;
```

### `.output()`

* Defines the [output format](https://developers.cloudflare.com/images/transform-images/) for the transformed image such as AVIF, WebP, and JPEG.
* Image [quality](https://developers.cloudflare.com/images/transform-images/transform-via-url/#quality) can be altered by specifying `quality` on a 1-100 scale.

For example, to rotate, resize, and blur an image, then output the image as AVIF:

```ts
const info = await env.IMAGES.info(stream);
// stream contains a valid image, and width/height is available on the info object


const response = (
  await env.IMAGES.input(stream)
    .transform({ rotate: 90 })
    .transform({ width: 128 })
    .transform({ blur: 20 })
    .output({ format: "image/avif" })
).response();


return response;
```

### `.info()`

* Outputs information about the image, such as `format`, `fileSize`, `width`, and `height`.

Responses from the Images binding are not automatically cached. Workers lets you interact directly with the Cache API to customize cache behavior using Workers. You can implement logic in your script to store transformations in Cloudflare’s cache.

## Interact with your Images binding locally

The Images API can be used in local development through [Wrangler](https://developers.cloudflare.com/workers/wrangler/install-and-update/), the command-line interface for Workers. Using the Images binding in local development will not incur usage charges.

Wrangler supports two different versions of the Images API:

* A high-fidelity version that supports all features that are available through the Images API. This is the same version that Cloudflare runs globally in production.
* A low-fidelity offline version that supports only a subset of features, such as resizing and rotation.

To test the high-fidelity version of Images, you can run `wrangler dev`:

```txt
npx wrangler dev
```

This creates a local-only environment that mirrors the production environment where Cloudflare runs the Images API. You can test your Worker with all available transformation features before deploying to production.

To test the low-fidelity offline version of Images, add the `--experimental-images-local-mode` flag:

```txt
npx wrangler dev --experimental-images-local-mode
```

Currently, this version supports only `width`, `height`, `rotate`, and `format`.

When testing with the [Workers Vitest integration](https://developers.cloudflare.com/workers/testing/vitest-integration/), the low-fidelity offline version is used by default, to avoid hitting the Cloudflare API in tests.
